/****************************************************************
 Title:     LCD Library 
 Author:    Mahmood Alimohammadi
 
 Hardware:    AVR ATmega Microcontrollers
 Software: AVR GCC
 Compiler: Atmel Studio 6
 Version 1
 Date: September 24, 2020
 
DESCRIPTION
This library is written in C and uses 4-bit IO port mode to 
interface with alphanumeric LCDs based on HD44780U. It 
does NOT support 8-bit IO port mode
 
RW pin of the LCD module should be connected to GROUND.

Supports 16x1, 16x2, 16x4, and 20x4 Alphanumeric LCDs
       
*****************************************************************/
/* 
In Atmel studio 6, go to Project menu -->Project Name Properties--> 
* Toolchain --> AVR/GNU C Compiler --> Symbols
* Add F_CPU=1000000 (or whatever) to Defined Symbols(-D)
*/

#ifndef	F_CPU 1000000
#define F_CPU 1000000
#endif

#include <avr/io.h>		//used for ports
#include <util/delay.h>	//used for delays
#include <stdio.h>		//used for itoa
#include <stdlib.h>		//used for functions		
#include <stdint.h>		//used for uint8_t

#include "lcd.h"

/* Address of PORT(X)= Address of DDR(X)+1 */
#define LCD_PORT(X) (*(&X+1))	
/*******************************************
FUNCTIONS
 
Name: LCD_Text
Print a string of characters on the display
Parameter: string-->string of characters to be displayed
Return: None
****/
void LCD_Text(char *string)
	{
		uint8_t c;
		c= 0;
		while(string[c]!=0)
			{
				LCD_Chr(string[c]);
				c++;
				_delay_us(50);
			}

	}
/*------------------------------------*/	
/**
Name: LCD_Int
Print an integer number on the display
Parameter: number-->the intiger number to be displayed
Return: None
****/

void LCD_Int(int number)
{
	char value[10];
	itoa(number, value, 10); //Here 10 means decimal
	LCD_Text(value);
}
/*------------------------------------*/
/*
To print float numbers in Atmel Studio 6, or 7, 
go to Project Name Properties-->Toolchain-->
AVR/GNU Linker->General and select 
 -Wl,-u,vfprintf
Then click Miscellaneous under the AVR/GNU Linker and 
add the following in the Other Linker Flags box :
-lprintf_flt 
*/

/**
Name: LCD_Float
Print a floating point number on the display
with or without decimal digits
Parameters: f-->the float number to be displayed
			d-->number of decimal digits. 
e.g. d=2=two decimal digits, d=0= no decimal digit
Return: None
****/
void LCD_Float(double f, uint8_t d)
{
	char buffer[10];
	switch(d)
		{
			/* 0 decimal digit*/
			case 0:	sprintf(buffer, "%.0f", f); 
			LCD_Text(buffer);break;
			/*1 decimal digit*/
			case 1:	sprintf(buffer, "%.1f", f); 
			LCD_Text(buffer);break;
			/*2 decimal digits*/
			case 2:	sprintf(buffer, "%.2f", f); 
			LCD_Text(buffer);break;
			/*3 decimal digits*/
			case 3:	sprintf(buffer, "%.3f", f);
			LCD_Text(buffer);break;
			/*4 decimal digits*/
			case 4:	sprintf(buffer, "%.4f", f);
			LCD_Text(buffer);break;
			/*5 decimal digits*/
			case 5:	sprintf(buffer, "%.5f", f);
			LCD_Text(buffer);break;
	}
	
}
/*------------------------------------*/
/**
 Name: LCD_Custom_chr
 Generate a custom character 
 Parameters: l-->the location of the character
			 cc--> to be generated
 Return: None
 ****/

void LCD_Custom_chr (uint8_t l, char *cc)
{
	uint8_t i;
	if(l<8)
	{
		LCD_cmnd(0x40 + (l*8)); 
		for(i=0;i<8;i++) 
		//store custom character in CGRAM 
		LCD_Chr(cc[i]);
		
	}
}
/*------------------------------------*/
/**
Name: LCD_Chr
Send character to the display
Parameter: character-->to be sent to the display
Return: None
****/
void LCD_Chr( char character)
	{
		LCD_Reset_Bits();
		LCD_High_Bits(character);
		LCD_RS_EN_Data_Cmnd();
		LCD_Reset_Bits();
		LCD_Low_Bits(character);
		LCD_RS_EN_Data_Cmnd();
	}
/*------------------------------------*/
/**
Name: LCD_cmnd
Send commands to the display
Parameter: cmnd-->to be sent to the display
Return: None
****/
        
void LCD_cmnd( char cmnd)
	{
		
		LCD_Reset_Bits();
		LCD_High_Bits(cmnd);
		LCD_RS_EN_Cntrl_Cmnd();
		LCD_Reset_Bits();
		LCD_Low_Bits(cmnd);
		LCD_RS_EN_Cntrl_Cmnd();
	}
/*------------------------------------*/
/**
Name: LCD_goto
Set cursor to the specified position
Parameters: y-->line position
		    x--> character position
			x and y start from 1 

Return:None
****/
void LCD_goto(uint8_t y, uint8_t x)
	{	
		
		if(lcd_lines==1 && lcd_characters==16)
			{
				uint8_t value[1] = {line_1}; //16X1 LCD
				LCD_cmnd(value[y-1] + x-1);
				_delay_us(400);		
			}		
		else if(lcd_lines==2 && lcd_characters==16)
			{
				 uint8_t value[2] = {line_1,line_2}; //16X2 LCD
				LCD_cmnd(value[y-1] + x-1);
				_delay_us(400);
		
			}								
		else if(lcd_lines==4 && lcd_characters==16)
			{	
				uint8_t value[4] = {line_1, line_2, line_3,line_4}; //16X4 LCD		
				LCD_cmnd(value[y-1] + x-1);
				_delay_us(400);
		}
		else if	(lcd_lines==4 && lcd_characters==20)
			{
				uint8_t value[4] = {line_1,line_2, line3,line4}; //20X4 LCD
				LCD_cmnd(value[y-1] + x-1);
				_delay_us(400);
			}
			
	}											
 /*------------------------------------*/
/**
Name: LCD_Clear
Clear entire display
Cursor at home position 
Parameter: None
Return: None
****/
void LCD_Clear(void)
	{
		LCD_cmnd(lcd_clean);    /* Clear display */
		_delay_ms(10);
		LCD_cmnd(lcd_ddram);    /* Cursor at home position */
	}
/*------------------------------------*/	 
/**
Name: LCD_Clr_Line
Clear the specified line(1,2,3, or 4) on the display
Parameter: l--> line number
Return: None
****/

void LCD_Clr_Line(uint8_t l)
{
	char i;
	if(lcd_characters==16)
	{
		
		for(i=1;i<17;i++)
		{
			LCD_goto(l,i);
			LCD_Text(" ");
		}
	}
	
	else if(lcd_characters==20)
	{
		for(i=1;i<21;i++)
		{
			LCD_goto(l,i);
			LCD_Text(" ");
		}
	}
}
/*------------------------------------*/
/**
Name: LCD_RS_EN_Cntrl_Cmnd
Commands to Clear RS first and then Set and
Clear EN after a delay
Parameter: None
Return: None
****/
void LCD_RS_EN_Cntrl_Cmnd(void)
	{
		/* RS low: Select the Command Register */
		LCD_PORT(lcd_rs_ddr) &= ~(1<<lcd_rs_bit); 
		LCD_PORT(lcd_en_ddr) |= (1<<lcd_en_bit);
		_delay_us(5);
		LCD_PORT(lcd_en_ddr) &= ~(1<<lcd_en_bit);
		_delay_us(400);
	}
/*------------------------------------*/
/**
Name: LCD_RS_EN_Data_Cmnd
Commands to Set RS  to write data first and 
then Set and Clear EN after a delay
Parameter: None
Return: None
****/
void LCD_RS_EN_Data_Cmnd(void)
	{
		/* RS high: Select the Data Register. */    
		LCD_PORT(lcd_rs_ddr) |= (1<<lcd_rs_bit);       
		LCD_PORT(lcd_en_ddr) |= (1<<lcd_en_bit); 
		_delay_us(5);
		LCD_PORT(lcd_en_ddr) &= ~(1<<lcd_en_bit);
		_delay_us(400);
	}
/*------------------------------------*/ 
/**
Name: LCD_Rest_Bits
		Bit 7 low
		Bit 6 low
		Bit 5 low
		Bit 4 low
Parameter: None
Return: None
****/ 
 void LCD_Reset_Bits()
	{
		LCD_PORT(lcd_d7_ddr) &= ~_BV(lcd_d7_bit);
		LCD_PORT(lcd_d6_ddr) &= ~_BV(lcd_d6_bit);
		LCD_PORT(lcd_d5_ddr) &= ~_BV(lcd_d5_bit);
		LCD_PORT(lcd_d4_ddr) &= ~_BV(lcd_d4_bit);
	}
/*------------------------------------*/
 /**
 Name: LCD_High_Bits
 Send the high bits of the control commands 
 or data to the display
 Parameter: data--> to be sent to the display
 Return: None
 ****/
 void LCD_High_Bits(char data)
 	 	 
	{
		if(data & 0X80) LCD_PORT(lcd_d7_ddr)|= _BV(lcd_d7_bit);
		if(data & 0X40) LCD_PORT(lcd_d6_ddr)|= _BV(lcd_d6_bit);
		if(data & 0X20) LCD_PORT(lcd_d5_ddr)|= _BV(lcd_d5_bit);
		if(data & 0X10) LCD_PORT(lcd_d4_ddr)|= _BV(lcd_d4_bit);
	}
/*------------------------------------*/ 
/**
Name: LCD_Low_Bits
Send the low bits of the control command or
data to the display
Parameter: data to be sent to the display
Return:None
**/
void LCD_Low_Bits(char data)
	{
		if(data & 0X08) LCD_PORT(lcd_d7_ddr)|= _BV(lcd_d7_bit);
		if(data & 0X04) LCD_PORT(lcd_d6_ddr)|= _BV(lcd_d6_bit);
		if(data & 0X02) LCD_PORT(lcd_d5_ddr)|= _BV(lcd_d5_bit);
		if(data & 0X01) LCD_PORT(lcd_d4_ddr)|= _BV(lcd_d4_bit);
	}
/*------------------------------------*/
/**
Name: LCD_Scroll_Right
Scroll the string of characters to right
Parameter: None
Return: None
****/
void LCD_Scroll_Right(void)
	{
		LCD_cmnd(lcd_right_shift);    
		_delay_ms(100);  //Scroll speed delay
	
	}
/*------------------------------------*/	
/**
 Name: LCD_Scroll_Left
Scroll the string of characters to left
Parameter: None
Return: None
****/
void LCD_Scroll_Left(void)
	{
		LCD_cmnd(lcd_left_shift);    
		_delay_ms(100);  //Scroll speed delay
	}
/*------------------------------------*/
/**
 Name: LCD_Init
 Clear LCD, display off
 4bit(DL=0),2line(Bit1=N=1),5x7 pixel(4bit=bit4=0
 LCD on, Cursor off
 Parameter: None
 Return: None
****/
void LCD_init()
	{
		
		// Configure the data direction register as output
		lcd_d4_ddr |= (1<<lcd_d4_bit);
		lcd_d5_ddr |= (1<<lcd_d5_bit);
		lcd_d6_ddr |= (1<<lcd_d6_bit);
		lcd_d7_ddr |= (1<<lcd_d7_bit);
		
		// Configure the RS and EN pins as output
		lcd_rs_ddr |= (1<<lcd_rs_bit);
		lcd_en_ddr |= (1<<lcd_en_bit);
				
		LCD_PORT(lcd_rs_ddr) &= ~(1<<lcd_rs_bit);      // RS low
		LCD_PORT(lcd_en_ddr) &= ~(1<<lcd_en_bit);      // EN low
					
		
		/* Wait 50ms after power-on */
		_delay_ms(50);
		/* Function set (Data Length: 8 bits),1 of 3 */
		LCD_cmnd(lcd_DL8); 
		_delay_ms(10);
		/* Function set (Data Length: 8 bits), 2 of 3 */
		LCD_cmnd(lcd_DL8);	
		_delay_us(200);
		/* Function set (Data Length: 8 bits), 3 of 3 */
		LCD_cmnd(lcd_DL8);	
		_delay_us(200);
		/* Function set-Initialize LCD in 4-bit */
		LCD_cmnd(lcd_DL4_init);  
		_delay_us(100);
		
		/*Function set (Set interface to be 4 bits long) */
		LCD_cmnd(lcd_DL4); 
		_delay_us(100);
		
		LCD_cmnd(lcd_off); //Display off
		
		LCD_cmnd(lcd_clean); // Clear display
		_delay_ms(10);
		LCD_cmnd(cursor_shift_left); // Entry mode
								
		_delay_us(100);
		 LCD_on_Curs_on_off_blink();
		_delay_us(100);
	}
/*------------------------------------*/
/**
 Name: LCD_on_Curs_on_off_blink
 LCD_Cursor = 0 = Display ON, Cursor OFF
 LCD_Cursor = 1 = Display ON, Cursor ON
 LCD_Cursor = 2 = Display ON, Cursor Character Blink
 Default= Display ON, Cursor OFF	
 
 Parameter: None
 Return: None
****/	

void LCD_on_Curs_on_off_blink(void)
	{
		switch(LCD_Cursor)
			{
				/*Display ON, Cursor Off*/
				case 0: LCD_cmnd(lcd_on_curs_off); break; 
				/*Display ON, Cursor ON*/
				case 1:	LCD_cmnd(lcd_on_curs_on); break;
				/*Display ON, Cursor Character Blink*/	
				case 2:	LCD_cmnd(lcd_on_curs_blink); break;
				/*Display ON, Cursor OFF*/
				default: LCD_cmnd(lcd_on_curs_off); break; 
			}	
	}	
	
	